
#' Transform data to means of 0, standard deviations of 1 and zero Pearson correlations
#' 
#' @param data matrix with observations in rows and variables in columns (float or integer; defined on a metric space)
#' @param center Boolean: whether to set means to 0; default TRUE
#' @param scale Boolean: whether to set standard deviations to 1; default TRUE
#' @param rotate Boolean: whether to set Pearson correlations to 0 (by eigenvector decomposition); default TRUE
#' 
#' @return matrix of dimensions dim(data)
#' @export
#' 
#' @examples
#' normiris <- normalise(data=iris)


normalise <- function(data, center=TRUE, scale=TRUE, rotate=TRUE){
	# data is a matrix of real numbers, which does not include the dependent variable
	#   (if there is one) -- normalise() will also accept data.frame here
	# booleans center (note American-style spelling), scale, and rotate, turn on or off
	#   the following actions: 
	#   * translation so each dimension's mean is 0
	#   * scaling so each dimension's standard deviation is 1
	#   * rotation by the eigenvector matrix (of the covariance matrix) 
	#   * if (rotate & scale) then the rotated variables are scaled again

	# centering and scaling
	normdata <- scale(data, center=center, scale=scale)

	# store vectors and prepare matrix (NULL allows simple setting of all attributes)
	center_vector <- scale_vector_1 <- scale_vector_2 <- rotate_matrix <- NULL
	if(center) { center_vector <- attr(normdata,"scaled:center") }
	if(scale) { scale_vector_1  <- attr(normdata,"scaled:scale") }

	# rotation
	if(rotate) {
		eigen_decomp <- eigen(cov(normdata))
		rotate_matrix <- eigen_decomp$vectors
		attr(normdata, "scaled:center") <- NULL
		attr(normdata, "scaled:scale") <- NULL
		normdata <- normdata %*% rotate_matrix
		if(scale) {
			scale_vector_2 <- sqrt(eigen_decomp$values)
			normdata <- t(apply(normdata,1,function(z){z/scale_vector_2}))
		}
	}

	# store attributes so there can be back-transformation later
	attr(normdata, "center_vector") <- center_vector
	attr(normdata, "scale_vector_1") <- scale_vector_1
	attr(normdata, "scale_vector_2") <- scale_vector_2
	attr(normdata, "rotate_matrix") <- rotate_matrix

	return(normdata)

}

# accommodate American spelling
normalize <- normalise