# predict densities and marginal densities for kudzu







# This function has inputs: x (p vector), kudzu_list:

# proximity is the distance from a leaf face in units of kudzu_list$sigma, 
#   to regard as neighbourly -- indicated by phi in our math formulas
# The option use_all_leaves forces it to skip finding the neighbourhood and evaluate 
#   the density with all leaves
# tops and bottoms should already have been delta-translated if required
predict_joint_kudzu <- function(x, kudzu_list, proximity=6, use_all_leaves=FALSE){
	# reduce to only neighbour leaves, if desired
	if(!use_all_leaves){
		neighbourhood <- neighbour_leaves(x, kudzu_list, proximity)
	} else {
		neighbourhood <- 1:(length(kudzu_list$heights)) # otherwise, use all leaves
	}
	rtops <- kudzu_list$tops[neighbourhood, , drop=FALSE]
	rbottoms <- kudzu_list$bottoms[neighbourhood, , drop=FALSE]
	rheights <- kudzu_list$heights[neighbourhood]
	
	# get ramp values
	ramptops <- t(apply(rtops,1,function(t){expit((t-x)/kudzu_list$sigma)}))
	rampbottoms <- t(apply(rbottoms,1,function(b){expit((x-b)/kudzu_list$sigma)}))

	# multiply together for each leaf, with height
	leaf_contributions <- apply(ramptops,1,prod) * apply(rampbottoms,1,prod) * rheights

	# add to get kudzu density
	kudzu_sum <- sum(leaf_contributions)

	return(kudzu_sum)
}



#' Obtain density at a point given a kudzu density function
#' 
#' @param x numeric vector at which to evaluate density; any NA elements will lead to those dimensions being marginalised out
#' @param kudzu_list a kudzu list, such as kudzu::tree2kudzu() returns
#' 
#' @return positive real scalar
#' @export
#' 
#' @examples
#' predict_kudzu(x=c(-0.1, 0, 0.1, 0),
#' 	             kudzu_list=iriskudzu)
predict_kudzu <- function(x, kudzu_list) {
	L <- NROW(kudzu_list$tops)
	p <- NCOL(kudzu_list$tops)
	q <- sum(as.numeric(is.na(x))) 

	# make Lxp matrix which contains either integral (from kudzu_list$leaf_dim_integrals)
	#   or point density. Multiply p components together, multiply by 
	#   kudzu_list$det_densities[l] and you have the leaf contribution. 
	#   Add all of these together and divide by kudzu_list$kudzu_integral
	leaf_dim_factors <- matrix(NA_real_,nrow=L,ncol=p)
	for(j in 1:p){
		if(is.na(x[j])){
			leaf_dim_factors[,j] <- kudzu_list$leaf_dim_integrals[,j]
		}
		else{
			for(l in 1:L){
				# evaluate product of ramps without DET density
				leaf_dim_factors[l,j] <- expit((x[j]-kudzu_list$bottoms[l,j])/kudzu_list$sigma) *
				                          expit((kudzu_list$tops[l,j]-x[j])/kudzu_list$sigma)
			}
		}
	}
	leaf_terms <- apply(leaf_dim_factors,1,prod) * kudzu_list$det_densities
	total_density <- sum(leaf_terms) / kudzu_list$kudzu_integral

	return(total_density)
}



