# predict function for tree

# takes a current_tree list; intended for programmer use
predict_current_tree <- function(x,current_tree) {
	leaf <- search_tree(x,current_tree)
	return(current_tree$nodes[[leaf]]$predicted)
}

# takes a list out of store_tree(), x has no NAs
predict_leaves <- function(x, stored_tree) {
	return(stored_tree$densities[search_leaves(x,stored_tree)])
}


# predict marginal density; x is a p-vector with q NAs, 0 < q < p
predict_marg_leaves <- function(x, stored_tree) {
	# which dimensions are not specified (the q NAs)?
	subspace <- is.na(x)
	q <- sum(as.numeric(subspace))

	# which leaves cross the q-dimensional subspace?
	relevant_leaves <- search_leaves(x, stored_tree)

	# get details of just those leaves
	rtops <- stored_tree$tops[relevant_leaves, , drop=FALSE]
	rbottoms <- stored_tree$bottoms[relevant_leaves, , drop=FALSE]
	rdensities <- stored_tree$densities[relevant_leaves]

	# reduce to the q dimensions of interest (keep it a matrix)
	rtops <- rtops[,subspace, drop=FALSE]
	rbottoms <- rbottoms[,subspace, drop=FALSE]

	# integrate over the q dimensions
	widths <- rtops-rbottoms
	volume <- apply(widths, 1, prod) 
	marg_density_l <- rdensities * volume

	# sum the leaves
	return(sum(marg_density_l))  
}


#' Obtain density at a point given a DET
#' 
#' @param x numeric vector at which to evaluate density; any NA elements will lead to those dimensions being marginalised out
#' @param stored_tree a tree list, such as kudzu::tree() returns
#' 
#' @return positive real scalar
#' @export
#' 
#' @examples
#' predict_tree(x=c(-0.1, 0, 0.1, 0),
#' 	             stored_tree=irisdet)
predict_tree <- function(x, stored_tree) {
	p <- length(stored_tree$densities)
	q <- sum(as.numeric(is.na(x)))

	# choose joint density, marginal density, or (hard coded for 
	#   completeness, but can also use predict_marg_leaves() for this) 1
	switch_q <- 1 + as.numeric(q>0) + as.numeric(q==p)
	return(switch(switch_q,
		          predict_leaves(x, stored_tree),
		          predict_marg_leaves(x, stored_tree),
		          1))
}

# get log likelihood of tree given data
tree_loglik <- function(current_tree, data) {
	return(log(apply(data,1,predict_current_tree,current_tree=current_tree)))
}