#' RNG function given a kudzu density function
#' 
#' @param n integer greater than 0: number of values to return
#' @param kudzu_list a kudzu list, such as kudzu::tree2kudzu() returns
#' 
#' @return matrix with n rows and NCOL(kudzu_list$tops) columns
#' @export
#' 
#' @examples
#' rkudzu(100, kudzu_list=iriskudzu)
rkudzu <- function(n,kudzu_list) {
	L <- nrow(kudzu_list$tops)
	p <- ncol(kudzu_list$tops)

	# get probability of each leaf
	leaf_probs <- kudzu_list$leaf_integrals / kudzu_list$kudzu_integral

	# pick leaves
	leaf_id <- sample(1:L, replace=TRUE, size=n, prob=leaf_probs)

	# make matrix to hold results
	output <- matrix(NA_real_, nrow=n, ncol=p)

	# fill matrix with uniform random numbers
	for(i in 1:n) {
		for(j in 1:p){
			output[i,j] <- runif(n=1,
				                 min=kudzu_list$bottoms[leaf_id[i],j],
				                 max=kudzu_list$tops[leaf_id[i],j])
		}
	}

	# for each dimension, reflect around the top/bottom according to ramp probability
	for(i in 1:n) {
		for(j in 1:p){
			# nearer to top or bottom?
			distance_below_top <- kudzu_list$tops[leaf_id[i],j]-output[i,j]
			distance_above_bottom <- output[i,j]-kudzu_list$bottoms[leaf_id[i],j]
			# if nearer to top ...
			if(distance_above_bottom > distance_below_top){
				reflect_prob <- 1 - (1 / (1+exp(-(distance_below_top/kudzu_list$sigma))))
				if(runif(n=1,min=0,max=1) < reflect_prob){
					output[i,j] <- kudzu_list$tops[leaf_id[i],j]+distance_below_top
				}
			}
			# if nearer to bottom ...
			else{
				reflect_prob <- 1 - (1 / (1+exp(-(distance_above_bottom/kudzu_list$sigma))))
				if(runif(n=1,min=0,max=1) < reflect_prob){
					output[i,j] <- kudzu_list$bottoms[leaf_id[i],j]-distance_above_bottom
				}
			}
		}
	}

	return(output)
}